<?php
// Gallery.php - automatikus képgaléria optimalizációval
$gallery_dir = 'gallery/';
$optimized_dir = 'gallery/optimized/';
$allowed_extensions = ['jpg', 'jpeg', 'png', 'webp'];
$images = [];

// Optimalizált mappa létrehozása
if (!is_dir($optimized_dir)) {
    mkdir($optimized_dir, 0755, true);
}

// Kép optimalizáló függvény
function optimizeImage($source_path, $optimized_path, $max_dimension = 1000, $max_filesize = 500000) {
    // Ha már létezik az optimalizált verzió, nem kell újra csinálni
    if (file_exists($optimized_path)) {
        return $optimized_path;
    }
    
    // Eredeti fájl mérete és dimenziói
    $file_size = filesize($source_path);
    $image_info = getimagesize($source_path);
    
    if (!$image_info) return $source_path;
    
    $width = $image_info[0];
    $height = $image_info[1];
    $type = $image_info[2];
    
    // Ha kicsi a kép és kis fájlméret, nem kell optimalizálni
    if ($width <= $max_dimension && $height <= $max_dimension && $file_size <= $max_filesize) {
        copy($source_path, $optimized_path);
        return $optimized_path;
    }
    
    // Kép betöltése típus szerint
    switch ($type) {
        case IMAGETYPE_JPEG:
            $source_image = imagecreatefromjpeg($source_path);
            break;
        case IMAGETYPE_PNG:
            $source_image = imagecreatefrompng($source_path);
            break;
        case IMAGETYPE_WEBP:
            $source_image = imagecreatefromwebp($source_path);
            break;
        default:
            return $source_path;
    }
    
    if (!$source_image) return $source_path;
    
    // Új méretek kiszámítása (arány megtartásával)
    $new_width = $width;
    $new_height = $height;
    
    if ($width > $max_dimension || $height > $max_dimension) {
        if ($width > $height) {
            $new_width = $max_dimension;
            $new_height = round(($height * $max_dimension) / $width);
        } else {
            $new_height = $max_dimension;
            $new_width = round(($width * $max_dimension) / $height);
        }
    }
    
    // Új kép létrehozása
    $optimized_image = imagecreatetruecolor($new_width, $new_height);
    
    // PNG átlátszóság megőrzése
    if ($type == IMAGETYPE_PNG) {
        imagealphablending($optimized_image, false);
        imagesavealpha($optimized_image, true);
        $transparent = imagecolorallocatealpha($optimized_image, 255, 255, 255, 127);
        imagefill($optimized_image, 0, 0, $transparent);
    }
    
    // Kép átméretezése
    imagecopyresampled($optimized_image, $source_image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
    
    // Mentés minőség beállításokkal
    $quality = 85;
    if ($file_size > $max_filesize) {
        $quality = 75; // Nagyobb tömörítés nagy fájloknál
    }
    
    $saved = false;
    switch ($type) {
        case IMAGETYPE_JPEG:
            $saved = imagejpeg($optimized_image, $optimized_path, $quality);
            break;
        case IMAGETYPE_PNG:
            $saved = imagepng($optimized_image, $optimized_path, 6); // PNG tömörítés 0-9
            break;
        case IMAGETYPE_WEBP:
            $saved = imagewebp($optimized_image, $optimized_path, $quality);
            break;
    }
    
    // Memória felszabadítása
    imagedestroy($source_image);
    imagedestroy($optimized_image);
    
    return $saved ? $optimized_path : $source_path;
}

// Képek betöltése és optimalizálása
if (is_dir($gallery_dir)) {
    $files = scandir($gallery_dir);
    foreach ($files as $file) {
        if ($file != '.' && $file != '..' && $file != 'optimized') {
            $file_extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            if (in_array($file_extension, $allowed_extensions)) {
                $source_path = $gallery_dir . $file;
                $optimized_path = $optimized_dir . $file;
                
                // Kép optimalizálása
                $final_path = optimizeImage($source_path, $optimized_path);
                $images[] = $final_path;
            }
        }
    }
    // Képek rendezése név szerint
    sort($images);
}
?>

<!-- Munkáink szekció -->
<section class="py-20 bg-white">
    <div class="container mx-auto px-4">
        <div class="text-center mb-16">
            <h2 class="text-4xl md:text-5xl font-bold text-tertiary mb-6">Munkáink galéria</h2>
            <p class="text-xl text-gray-600 max-w-3xl mx-auto">
                Tekintse meg eddigi projektjeinket! Lakatosmunkáktól a komplex fémszerkezetekig - 
                minden munkánk a minőség és precizitás jegyében született.
            </p>
        </div>
        
        <?php if (!empty($images)): ?>
            <!-- Képgaléria rács -->
            <div class="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6" id="gallery-grid">
                <?php foreach ($images as $index => $image): ?>
                    <div class="gallery-item group cursor-pointer overflow-hidden rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2" 
                         data-index="<?php echo $index; ?>" 
                         onclick="openLightbox(<?php echo $index; ?>)">
                        <div class="relative aspect-square">
                            <img src="<?php echo $image; ?>" 
                                 alt="BDSTEEL munkák - lakatosmunkák, lépcsőépítés, fémszerkezetek" 
                                 class="w-full h-full object-cover transition-transform duration-300 group-hover:scale-110"
                                 loading="lazy">
                            <div class="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-30 transition-all duration-300 flex items-center justify-center">
                                <div class="opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                                    <i class="fas fa-search-plus text-white text-2xl"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
            
            <!-- Lightbox -->
            <div id="lightbox" class="fixed inset-0 bg-black bg-opacity-90 z-50 hidden flex items-center justify-center">
                <div class="relative max-w-7xl max-h-full mx-4">
                    <!-- Bezárás gomb -->
                    <button onclick="closeLightbox()" class="absolute -top-12 right-0 text-white hover:text-accent text-2xl transition-colors duration-300">
                        <i class="fas fa-times"></i>
                    </button>
                    
                    <!-- Előző gomb -->
                    <button onclick="previousImage()" class="absolute left-4 top-1/2 transform -translate-y-1/2 text-white hover:text-accent text-3xl transition-colors duration-300 bg-black bg-opacity-50 rounded-full w-12 h-12 flex items-center justify-center">
                        <i class="fas fa-chevron-left"></i>
                    </button>
                    
                    <!-- Következő gomb -->
                    <button onclick="nextImage()" class="absolute right-4 top-1/2 transform -translate-y-1/2 text-white hover:text-accent text-3xl transition-colors duration-300 bg-black bg-opacity-50 rounded-full w-12 h-12 flex items-center justify-center">
                        <i class="fas fa-chevron-right"></i>
                    </button>
                    
                    <!-- Kép -->
                    <img id="lightbox-image" src="" alt="" class="max-w-full max-h-full rounded-lg shadow-2xl">
                    
                    <!-- Kép számláló -->
                    <div class="absolute -bottom-12 left-1/2 transform -translate-x-1/2 text-white text-center">
                        <span id="image-counter"></span>
                    </div>
                </div>
            </div>
            
        <?php else: ?>
            <!-- Üres galéria -->
            <div class="text-center py-16">
                <div class="bg-gray-100 rounded-3xl p-12 max-w-md mx-auto">
                    <i class="fas fa-images text-gray-400 text-6xl mb-6"></i>
                    <h3 class="text-2xl font-bold text-gray-600 mb-4">Galéria frissítés alatt</h3>
                    <p class="text-gray-500">
                        Munkáink képei hamarosan elérhetőek lesznek. 
                        Addig is tekintse meg szolgáltatásainkat vagy vegye fel velünk a kapcsolatot!
                    </p>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>

<script>
// Lightbox változók
let currentImageIndex = 0;
const images = <?php echo json_encode($images); ?>;

// Lightbox megnyitása
function openLightbox(index) {
    currentImageIndex = index;
    updateLightboxImage();
    document.getElementById('lightbox').classList.remove('hidden');
    document.body.style.overflow = 'hidden';
}

// Lightbox bezárása
function closeLightbox() {
    document.getElementById('lightbox').classList.add('hidden');
    document.body.style.overflow = 'auto';
}

// Következő kép
function nextImage() {
    currentImageIndex = (currentImageIndex + 1) % images.length;
    updateLightboxImage();
}

// Előző kép
function previousImage() {
    currentImageIndex = (currentImageIndex - 1 + images.length) % images.length;
    updateLightboxImage();
}

// Lightbox kép frissítése
function updateLightboxImage() {
    const lightboxImage = document.getElementById('lightbox-image');
    const imageCounter = document.getElementById('image-counter');
    
    lightboxImage.src = images[currentImageIndex];
    lightboxImage.alt = `BDSTEEL munkák ${currentImageIndex + 1}/${images.length}`;
    imageCounter.textContent = `${currentImageIndex + 1} / ${images.length}`;
}

// Billentyűzet események
document.addEventListener('keydown', function(e) {
    const lightbox = document.getElementById('lightbox');
    if (!lightbox.classList.contains('hidden')) {
        switch(e.key) {
            case 'Escape':
                closeLightbox();
                break;
            case 'ArrowLeft':
                previousImage();
                break;
            case 'ArrowRight':
                nextImage();
                break;
        }
    }
});

// Lightbox háttérre kattintás
document.getElementById('lightbox').addEventListener('click', function(e) {
    if (e.target === this) {
        closeLightbox();
    }
});

// Touch események mobil eszközökön
let touchStartX = 0;
let touchEndX = 0;

document.getElementById('lightbox').addEventListener('touchstart', function(e) {
    touchStartX = e.changedTouches[0].screenX;
});

document.getElementById('lightbox').addEventListener('touchend', function(e) {
    touchEndX = e.changedTouches[0].screenX;
    handleSwipe();
});

function handleSwipe() {
    const swipeThreshold = 50;
    const diff = touchStartX - touchEndX;
    
    if (Math.abs(diff) > swipeThreshold) {
        if (diff > 0) {
            nextImage(); // Balra swipe = következő kép
        } else {
            previousImage(); // Jobbra swipe = előző kép
        }
    }
}
</script>

<style>
/* Lightbox animációk */
#lightbox {
    backdrop-filter: blur(5px);
    animation: fadeIn 0.3s ease-out;
}

#lightbox-image {
    animation: scaleIn 0.3s ease-out;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes scaleIn {
    from { transform: scale(0.9); opacity: 0; }
    to { transform: scale(1); opacity: 1; }
}

/* Gallery item hover effekt */
.gallery-item:hover {
    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
}

/* Responsive lightbox gombok */
@media (max-width: 768px) {
    #lightbox button {
        font-size: 1.5rem;
        width: 3rem;
        height: 3rem;
    }
    
    #lightbox .absolute.left-4 {
        left: 1rem;
    }
    
    #lightbox .absolute.right-4 {
        right: 1rem;
    }
}
</style>